<?php
require 'db.php';

/** Usa $conexao ou $mysqli, conforme existir no db.php */
$cx = isset($conexao) ? $conexao : (isset($mysqli) ? $mysqli : null);
if (!$cx) { http_response_code(500); die("Conexão com o banco não encontrada."); }

function h($s){ return htmlspecialchars($s ?? '', ENT_QUOTES, 'UTF-8'); }

// ===== AÇÕES =====
if ($_SERVER["REQUEST_METHOD"] === "POST" && isset($_POST['acao'])) {
    switch ($_POST['acao']) {
        case 'adicionar_conta':
            $nome_conta = trim($_POST['nome_conta'] ?? '');
            $descricao  = trim($_POST['descricao'] ?? '');
            $valor_in   = str_replace(',', '.', $_POST['valor'] ?? '0');
            $valor      = is_numeric($valor_in) ? (float)$valor_in : 0;
            $data_venc  = $_POST['data_vencimento'] ?? '';
            $observ     = trim($_POST['observacoes'] ?? '');

            if ($nome_conta !== '' && $valor > 0 && $data_venc !== '') {
                $desc_final = $nome_conta . ($descricao ? (' - ' . $descricao) : '');
                $sql = "INSERT INTO transacoes (data_transacao, tipo, descricao, valor, origem, status)
                        VALUES (?, 'Despesa', ?, ?, ?, 'pendente')";
                $stmt = $cx->prepare($sql);
                $stmt->bind_param('ssds', $data_venc, $desc_final, $valor, $observ);
                if ($stmt->execute()) { header("Location: contas.php?status=success"); exit(); }
            }
            header("Location: contas.php?status=error"); exit();

        case 'marcar_pago':
            $id = (int)($_POST['id'] ?? 0);
            $data_pagamento = $_POST['data_pagamento'] ?? date('Y-m-d');
            $forma_pagamento = trim($_POST['forma_pagamento'] ?? '');
            if ($id > 0) {
                $stmt = $cx->prepare("UPDATE transacoes SET status='pago', data_pagamento=?, forma_pagamento=? WHERE id=? AND tipo='Despesa'");
                $stmt->bind_param('ssi', $data_pagamento, $forma_pagamento, $id);
                $stmt->execute();
                header("Location: contas.php?status=paid"); exit();
            }
            header("Location: contas.php?status=error"); exit();

        case 'excluir_conta':
            $id = (int)($_POST['id'] ?? 0);
            if ($id > 0) {
                $stmt = $cx->prepare("DELETE FROM transacoes WHERE id=? AND tipo='Despesa'");
                $stmt->bind_param('i', $id);
                if ($stmt->execute()) { header("Location: contas.php?status=deleted"); exit(); }
            }
            header("Location: contas.php?status=error"); exit();

        case 'limpar_todas_contas':
            if (($_POST['confirmar_limpeza'] ?? '') === 'SIM') {
                $cx->query("DELETE FROM transacoes WHERE tipo='Despesa'");
                header("Location: contas.php?status=all_deleted"); exit();
            }
            header("Location: contas.php?status=error"); exit();
    }
}

// Atualiza automaticamente despesas vencidas
$cx->query("UPDATE transacoes SET status='vencido' WHERE tipo='Despesa' AND status='pendente' AND data_transacao < CURDATE()");

// ===== FILTROS =====
$filtro = $_GET['filtro'] ?? 'todas';
$where  = "WHERE tipo='Despesa'";
switch ($filtro) {
    case 'pendentes':
        $where .= " AND status='pendente'";
        break;
    case 'vencidas':
        $where .= " AND status='vencido'";
        break;
    case 'pagas':
        $where .= " AND status='pago'";
        break;
    case 'vencendo':
        $where .= " AND status='pendente' AND data_transacao BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 7 DAY)";
        break;
    case 'todas':
    default:
        break;
}

$sql_contas = "SELECT id, descricao, valor, data_transacao, status, data_pagamento, forma_pagamento, origem
               FROM transacoes $where
               ORDER BY
                 CASE status WHEN 'vencido' THEN 0 WHEN 'pendente' THEN 1 WHEN 'pago' THEN 2 ELSE 3 END,
                 data_transacao ASC, id DESC";
$contas = $cx->query($sql_contas);

// Estatísticas
$stats = $cx->query("SELECT
    SUM(CASE WHEN tipo='Despesa' AND status='pendente' THEN 1 ELSE 0 END) AS pendentes,
    SUM(CASE WHEN tipo='Despesa' AND status='vencido'  THEN 1 ELSE 0 END) AS vencidas,
    SUM(CASE WHEN tipo='Despesa' AND status='pago'     THEN 1 ELSE 0 END) AS pagas,
    SUM(CASE WHEN tipo='Despesa' AND status IN ('pendente','vencido') THEN valor ELSE 0 END) AS valor_pendente
  FROM transacoes")->fetch_assoc();

?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>📋 Controle de Contas - Gabinete</title>

<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
<link rel="stylesheet" href="css/styles.css">


</head>
<body>

<div class="header">
  <div class="header-content">
    <div class="title-section">
      <h1><i class="fas fa-list-check"></i>Controle de Contas</h1>
      <p>Gerenciamento Completo de Transações</p>
    </div>
  </div>
</div>

<div class="container">
  <!-- NAV topo com as mesmas cores do index.php + Home (ícone) à direita -->
  <div class="navigation">
    <a href="index.php" class="nav-link nav-primary">
      <i class="fas fa-plus-circle"></i>Lançar Contas
    </a>
    <a href="contas.php" class="nav-link nav-green active">
      <i class="fas fa-list-check"></i>Controle de Contas
    </a>
    <a href="relatorios.php" class="nav-link nav-amber">
      <i class="fas fa-chart-bar"></i>Relatórios
    </a>
    <a href="http://localhost:8888/gabinete/" class="nav-link nav-primary nav-push-right" title="Voltar ao Sistema Principal" aria-label="Home">
      <i class="fas fa-home"></i>Sistema Principal
    </a>
  </div>

  <?php if (isset($_GET['status'])): ?>
    <?php if ($_GET['status'] === 'success'): ?><div class="alert alert-success">Conta adicionada como DESPESA (pendente)!</div><?php endif; ?>
    <?php if ($_GET['status'] === 'paid'): ?><div class="alert alert-success">Conta marcada como PAGA!</div><?php endif; ?>
    <?php if ($_GET['status'] === 'deleted'): ?><div class="alert alert-warning">Conta excluída.</div><?php endif; ?>
    <?php if ($_GET['status'] === 'all_deleted'): ?><div class="alert alert-warning">Todas as despesas foram removidas.</div><?php endif; ?>
    <?php if ($_GET['status'] === 'error'): ?><div class="alert alert-error">Erro ao processar. Tente novamente.</div><?php endif; ?>
  <?php endif; ?>

  <!-- Cards -->
  <div class="stats">
    <div class="stat-box"><h3>Pendentes</h3><p class="pendente"><?= (int)($stats['pendentes'] ?? 0) ?></p></div>
    <div class="stat-box"><h3>Vencidas</h3><p class="vencida"><?= (int)($stats['vencidas'] ?? 0) ?></p></div>
    <div class="stat-box"><h3>Pagas</h3><p class="paga"><?= (int)($stats['pagas'] ?? 0) ?></p></div>
    <div class="stat-box"><h3>Valor Pendente</h3><p class="pendente">R$ <?= number_format((float)($stats['valor_pendente'] ?? 0), 2, ',', '.') ?></p></div>
  </div>

  <!-- Formulário -->
  <div class="form-container">
    <h2>Nova Conta a Pagar</h2>
    <form method="post">
      <input type="hidden" name="acao" value="adicionar_conta">
      <div class="form-row">
        <div>
          <label for="nome_conta">Nome da Conta*</label>
          <input type="text" id="nome_conta" name="nome_conta" required>
        </div>
        <div>
          <label for="valor">Valor (R$)*</label>
          <input type="text" id="valor" name="valor" placeholder="150,50" required>
        </div>
        <div>
          <label for="data_vencimento">Data de Vencimento*</label>
          <input type="date" id="data_vencimento" name="data_vencimento" required>
        </div>
      </div>
      <div class="form-row">
        <div style="grid-column: span 2;">
          <label for="descricao">Descrição</label>
          <input type="text" id="descricao" name="descricao">
        </div>
        <div>
          <label for="observacoes">Observações</label>
          <textarea id="observacoes" name="observacoes" rows="3"></textarea>
        </div>
      </div>
      <button type="submit" class="btn-green">Adicionar Conta</button>
    </form>
  </div>

  <!-- Filtros -->
  <div class="filters">
    <a href="contas.php?filtro=todas" class="filter-btn <?= $filtro=='todas'?'active':'' ?>">Todas</a>
    <a href="contas.php?filtro=pendentes" class="filter-btn <?= $filtro=='pendentes'?'active':'' ?>">Pendentes</a>
    <a href="contas.php?filtro=vencendo" class="filter-btn <?= $filtro=='vencendo'?'active':'' ?>">Vencendo (7 dias)</a>
    <a href="contas.php?filtro=vencidas" class="filter-btn <?= $filtro=='vencidas'?'active':'' ?>">Vencidas</a>
    <a href="contas.php?filtro=pagas" class="filter-btn <?= $filtro=='pagas'?'active':'' ?>">Pagas</a>
  </div>

  <!-- Tabela -->
  <table>
    <thead>
      <tr><th>Descrição</th><th>Valor</th><th>Vencimento</th><th>Status</th><th>Pagamento</th><th>Ações</th></tr>
    </thead>
    <tbody>
      <?php while ($conta = $contas->fetch_assoc()): ?>
      <tr>
        <td><strong><?= h($conta['descricao']) ?></strong><br><small><?= h($conta['origem']) ?></small></td>
        <td><strong>R$ <?= number_format((float)$conta['valor'], 2, ',', '.') ?></strong></td>
        <td><?= date("d/m/Y", strtotime($conta['data_transacao'])) ?></td>
        <td><?= h(ucfirst($conta['status'] ?? '-')) ?></td>
        <td>
          <?php if (($conta['status'] ?? '') === 'pago'): ?>
            <?= !empty($conta['data_pagamento']) ? date("d/m/Y", strtotime($conta['data_pagamento'])) : '-' ?><br>
            <small><?= h($conta['forma_pagamento'] ?? '-') ?></small>
          <?php else: ?>
            <form method="post" style="display:inline" onsubmit="return confirm('Marcar como pago?');">
              <input type="hidden" name="acao" value="marcar_pago">
              <input type="hidden" name="id" value="<?= (int)$conta['id'] ?>">
              <input type="date" name="data_pagamento" value="<?= date('Y-m-d') ?>">
              <input type="text" name="forma_pagamento" placeholder="PIX/Cartão/..." style="width:110px">
              <button class="btn-small">Pagar</button>
            </form>
          <?php endif; ?>
        </td>
        <td>
          <form method="post" style="display:inline" onsubmit="return confirm('Excluir esta conta?');">
            <input type="hidden" name="acao" value="excluir_conta">
            <input type="hidden" name="id" value="<?= (int)$conta['id'] ?>">
            <button class="btn-small btn-danger">Excluir</button>
          </form>
        </td>
      </tr>
      <?php endwhile; ?>
      <?php if ($contas->num_rows === 0): ?>
        <tr><td colspan="6">Nenhum lançamento encontrado.</td></tr>
      <?php endif; ?>
    </tbody>
  </table>
</div>

</body>
</html>
