<?php
require_once __DIR__ . '/../models/Documento.php';
require_once __DIR__ . '/../helpers/AuthHelper.php';
require_once __DIR__ . '/../helpers/PermissionHelper.php';

/**
 * Controller dedicado para a Gestão de Documentos.
 */
class DocumentosController {

    /**
     * Exibe a página principal de gestão de documentos.
     */
    public function index() {
        AuthHelper::check();
        PermissionHelper::redirecionarSeNaoTemPermissao('documentos', 'visualizar');

        $documentoModel = new Documento();
        
        // Filtros da URL
        $categoria_filtro = $_GET['categoria'] ?? null;
        $tipo_midia_filtro = $_GET['tipo_midia'] ?? null;
        $busca = $_GET['busca'] ?? null;
        
        // Busca ou listagem
        if (!empty($busca)) {
            $documentos = $documentoModel->buscar($busca, $tipo_midia_filtro);
        } else {
            $documentos = $documentoModel->find($categoria_filtro, $tipo_midia_filtro);
        }
        
        $pageTitle = "Arquivo de Documentos e Multimídia";

        require_once __DIR__ . '/../views/partials/header.php';
        echo '<link rel="stylesheet" href="' . BASE_URL . '/public/css/documentos.css">';
        require_once __DIR__ . '/../views/documentos/index.php'; 
        require_once __DIR__ . '/../views/partials/footer.php';
    }

    /**
     * Processa o upload de um novo documento/mídia.
     */
    public function upload() {
        AuthHelper::check();
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['arquivo'])) {
            $titulo = trim($_POST['titulo']);
            $categoria = $_POST['categoria'];
            $descricao = trim($_POST['descricao'] ?? '');
            $tags = trim($_POST['tags'] ?? '');
            $evento_campanha = trim($_POST['evento_campanha'] ?? '');
            $arquivo = $_FILES['arquivo'];

            if ($arquivo['error'] !== UPLOAD_ERR_OK) {
                $_SESSION['upload_error'] = "Erro no upload do arquivo.";
                header('Location: ' . BASE_URL . '/documentos');
                exit();
            }

            $documentoModel = new Documento();
            
            // Determina o tipo de mídia
            $tipo_midia = $documentoModel->determinarTipoMidia($arquivo['name']);
            
            // Valida o arquivo
            $validacao = $documentoModel->validarArquivo($arquivo, $tipo_midia);
            if (!$validacao['valido']) {
                $_SESSION['upload_error'] = $validacao['erro'];
                header('Location: ' . BASE_URL . '/documentos');
                exit();
            }

            // Cria diretório baseado no tipo de mídia
            $uploadDir = 'uploads/documentos/' . $tipo_midia . 's/';
            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0777, true);
            }

            // Gera nome único para o arquivo
            $extensao = strtolower(pathinfo($arquivo['name'], PATHINFO_EXTENSION));
            $nomeArquivo = uniqid() . '_' . time() . '.' . $extensao;
            $caminhoArquivo = $uploadDir . $nomeArquivo;

            if (move_uploaded_file($arquivo['tmp_name'], $caminhoArquivo)) {
                // Dados básicos
                $dados = [
                    'id_usuario' => $_SESSION['usuario_id'],
                    'titulo' => $titulo,
                    'descricao' => $descricao,
                    'nome_arquivo' => $nomeArquivo,
                    'caminho_arquivo' => $caminhoArquivo,
                    'tipo_arquivo' => $arquivo['type'],
                    'tamanho_arquivo' => $arquivo['size'],
                    'categoria' => $categoria,
                    'tipo_midia' => $tipo_midia,
                    'tags' => $tags,
                    'evento_campanha' => $evento_campanha,
                    'largura' => null,
                    'altura' => null,
                    'duracao' => null,
                    'thumbnail' => null
                ];
                
                // Informações específicas para imagens
                if ($tipo_midia === 'imagem') {
                    $infoImagem = $documentoModel->obterInfoImagem($caminhoArquivo);
                    $dados['largura'] = $infoImagem['largura'];
                    $dados['altura'] = $infoImagem['altura'];
                }
                
                // Para vídeos, poderia implementar extração de thumbnail e duração aqui
                // if ($tipo_midia === 'video') { ... }
                
                if ($documentoModel->create($dados)) {
                    $tipoTexto = $tipo_midia === 'imagem' ? 'Imagem' : ($tipo_midia === 'video' ? 'Vídeo' : 'Documento');
                    $_SESSION['upload_success'] = "$tipoTexto enviado com sucesso!";
                } else {
                    $_SESSION['upload_error'] = "Erro ao salvar informações no banco de dados.";
                    // Remove o arquivo se falhou ao salvar no BD
                    if (file_exists($caminhoArquivo)) {
                        unlink($caminhoArquivo);
                    }
                }

            } else {
                $_SESSION['upload_error'] = "Erro ao mover o arquivo enviado.";
            }
        }
        
        header('Location: ' . BASE_URL . '/documentos');
        exit();
    }
    
    /**
     * Exibe estatísticas dos documentos
     */
    public function estatisticas() {
        AuthHelper::check();
        
        $documentoModel = new Documento();
        $estatisticas = $documentoModel->obterEstatisticas();
        
        header('Content-Type: application/json');
        echo json_encode($estatisticas);
        exit();
    }
}
