<?php
// Carrega os models e helpers necessários.
require_once __DIR__ . '/../models/Eleitor.php';
require_once __DIR__ . '/../helpers/AuthHelper.php';

/**
 * Controller para gerir as ações relacionadas a Eleitores.
 */
class EleitoresController {

    /**
     * Ação principal: Lista todos os eleitores ou busca por termo.
     */
    public function index() {
        AuthHelper::check(); 
        
        $eleitorModel = new Eleitor();
        
        // Verifica se há um termo de busca
        $busca = $_GET['busca'] ?? '';
        
        if (!empty($busca)) {
            $eleitores = $eleitorModel->search($busca);
        } else {
            $eleitores = $eleitorModel->getAll();
        }
        
        $pageTitle = "Lista de Eleitores";

        require_once __DIR__ . '/../views/partials/header.php';
        require_once __DIR__ . '/../views/eleitores/lista.php'; 
        require_once __DIR__ . '/../views/partials/footer.php';
    }

    /**
     * Exibe o formulário para criar um novo eleitor.
     * O nome do método foi alterado de 'create' para 'cadastrar' para corrigir o erro.
     */
    public function cadastrar() {
        AuthHelper::check();
        
        $pageTitle = "Cadastrar Novo Eleitor";

        require_once __DIR__ . '/../views/partials/header.php';
        // Carrega a view com o formulário de cadastro.
        require_once __DIR__ . '/../views/eleitores/create.php';
        require_once __DIR__ . '/../views/partials/footer.php';
    }

    /**
     * Salva um novo eleitor no banco de dados.
     * Esta função é chamada pelo formulário de 'create'.
     */
    public function store() {
        AuthHelper::check();

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $dados = $_POST;

            $eleitorModel = new Eleitor();
            if ($eleitorModel->create($dados)) {
                $_SESSION['success_message'] = "Eleitor cadastrado com sucesso!";
            } else {
                $_SESSION['error_message'] = "Erro ao cadastrar o eleitor. Tente novamente.";
            }
            
            header('Location: ' . BASE_URL . '/eleitores');
            exit();
        }
    }

    /**
     * Exibe os detalhes de um eleitor específico.
     */
    public function ver($id) {
        AuthHelper::check();
        
        $eleitorModel = new Eleitor();
        $eleitor = $eleitorModel->findById($id);
        
        if (!$eleitor) {
            $_SESSION['error_message'] = "Eleitor não encontrado.";
            header('Location: ' . BASE_URL . '/eleitores');
            exit();
        }
        
        $pageTitle = "Detalhes do Eleitor";
        
        require_once __DIR__ . '/../views/partials/header.php';
        require_once __DIR__ . '/../views/eleitores/detalhes.php';
        require_once __DIR__ . '/../views/partials/footer.php';
    }

    /**
     * Exibe o formulário para editar um eleitor.
     */
    public function editar($id) {
        AuthHelper::check();
        
        $eleitorModel = new Eleitor();
        $eleitor = $eleitorModel->findById($id);
        
        if (!$eleitor) {
            $_SESSION['error_message'] = "Eleitor não encontrado.";
            header('Location: ' . BASE_URL . '/eleitores');
            exit();
        }
        
        $pageTitle = "Editar Eleitor";
        
        require_once __DIR__ . '/../views/partials/header.php';
        require_once __DIR__ . '/../views/eleitores/editar.php';
        require_once __DIR__ . '/../views/partials/footer.php';
    }

    /**
     * Atualiza os dados de um eleitor.
     */
    public function atualizar($id) {
        AuthHelper::check();

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $dados = $_POST;

            $eleitorModel = new Eleitor();
            if ($eleitorModel->update($id, $dados)) {
                $_SESSION['success_message'] = "Eleitor atualizado com sucesso!";
            } else {
                $_SESSION['error_message'] = "Erro ao atualizar o eleitor. Tente novamente.";
            }
            
            header('Location: ' . BASE_URL . '/eleitores');
            exit();
        }
    }
}
