<?php

require_once __DIR__ . '/../models/Employee.php';
require_once __DIR__ . '/../helpers/AuthHelper.php';

class EmployeesController {
    private $employeeModel;
    
    public function __construct() {
        $this->employeeModel = new Employee();
    }
    
    /**
     * Exclusão em lote de colaboradores
     */
    public function deleteMultiple() {
        AuthHelper::check();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: ' . BASE_URL . '/employees');
            exit;
        }
        
        try {
            $ids = $_POST['employee_ids'] ?? [];
            
            if (empty($ids)) {
                $_SESSION['error'] = 'Nenhum colaborador selecionado para exclusão.';
                header('Location: ' . BASE_URL . '/employees');
                exit;
            }
            
            $result = $this->employeeModel->deleteMultiple($ids);
            
            if ($result) {
                $_SESSION['success'] = count($ids) . ' colaborador(es) excluído(s) com sucesso!';
            } else {
                $_SESSION['error'] = 'Erro ao excluir colaboradores.';
            }
        } catch (Exception $e) {
            $_SESSION['error'] = 'Erro ao excluir colaboradores: ' . $e->getMessage();
        }
        
        header('Location: ' . BASE_URL . '/employees');
        exit;
    }
    
    /**
     * Atualização em lote de colaboradores
     */
    public function updateMultiple() {
        AuthHelper::check();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: ' . BASE_URL . '/employees');
            exit;
        }
        
        try {
            $ids = $_POST['employee_ids'] ?? [];
            $updateData = [];
            
            if (empty($ids)) {
                $_SESSION['error'] = 'Nenhum colaborador selecionado para atualização.';
                header('Location: ' . BASE_URL . '/employees');
                exit;
            }
            
            // Capturar dados para atualização
            if (!empty($_POST['bulk_status'])) {
                $updateData['status'] = $_POST['bulk_status'];
            }
            if (!empty($_POST['bulk_department'])) {
                $updateData['department'] = $_POST['bulk_department'];
            }
            if (!empty($_POST['bulk_work_location'])) {
                $updateData['work_location'] = $_POST['bulk_work_location'];
            }
            
            if (empty($updateData)) {
                $_SESSION['error'] = 'Nenhum campo selecionado para atualização.';
                header('Location: ' . BASE_URL . '/employees');
                exit;
            }
            
            $result = $this->employeeModel->updateMultiple($ids, $updateData);
            
            if ($result) {
                $_SESSION['success'] = count($ids) . ' colaborador(es) atualizado(s) com sucesso!';
            } else {
                $_SESSION['error'] = 'Erro ao atualizar colaboradores.';
            }
        } catch (Exception $e) {
            $_SESSION['error'] = 'Erro ao atualizar colaboradores: ' . $e->getMessage();
        }
        
        header('Location: ' . BASE_URL . '/employees');
        exit;
    }
    
    /**
     * Buscar colaboradores por localização
     */
    public function searchByLocation() {
        AuthHelper::check();
        
        try {
            $latitude = $_GET['lat'] ?? null;
            $longitude = $_GET['lng'] ?? null;
            $radius = $_GET['radius'] ?? 10;
            
            if (!$latitude || !$longitude) {
                $_SESSION['error'] = 'Coordenadas de localização são obrigatórias.';
                header('Location: ' . BASE_URL . '/employees');
                exit;
            }
            
            $employees = $this->employeeModel->findByLocation($latitude, $longitude, $radius);
            $locationStats = $this->employeeModel->getLocationStats();
            
            require_once __DIR__ . '/../views/employees/index.php';
        } catch (Exception $e) {
            $_SESSION['error'] = 'Erro ao buscar por localização: ' . $e->getMessage();
            header('Location: ' . BASE_URL . '/employees');
            exit;
        }
    }
    
    /**
     * Exibir lista de colaboradores
     */
    public function index() {
        AuthHelper::check();
        
        try {
            // Capturar filtros da URL
            $filters = [
                'status' => $_GET['status'] ?? null,
                'name' => $_GET['search'] ?? null,
                'city' => $_GET['city'] ?? null,
                'state' => $_GET['state'] ?? null,
                'department' => $_GET['department'] ?? null,
                'work_location' => $_GET['work_location'] ?? null
            ];
            
            // Remover filtros vazios
            $filters = array_filter($filters, function($value) {
                return !empty($value);
            });
            
            // Buscar colaboradores com filtros
            if (!empty($filters)) {
                $employees = $this->employeeModel->search($filters);
            } else {
                $employees = $this->employeeModel->getAll();
            }
            
            // Obter estatísticas de localização
            $locationStats = $this->employeeModel->getLocationStats();
            
            require_once __DIR__ . '/../views/employees/index.php';
        } catch (Exception $e) {
            $_SESSION['error'] = 'Erro ao carregar colaboradores: ' . $e->getMessage();
            header('Location: ' . BASE_URL . '/rh');
            exit;
        }
    }
    
    /**
     * Exibir formulário de criação
     */
    public function create() {
        AuthHelper::check();
        require_once __DIR__ . '/../views/employees/create.php';
    }
    
    /**
     * Processar criação de colaborador
     */
    public function store() {
        AuthHelper::check();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: ' . BASE_URL . '/employees/create');
            exit;
        }
        
        try {
            // Validar dados obrigatórios
            $requiredFields = [
                'full_name', 'cpf', 'rg', 'birth_date', 'address', 'phone_number',
                'job_title', 'salary', 'hire_date', 'bank_name', 'agency_number', 'account_number'
            ];
            
            foreach ($requiredFields as $field) {
                if (empty($_POST[$field])) {
                    throw new Exception("Campo {$field} é obrigatório");
                }
            }
            
            // Validar CPF
            if (!$this->employeeModel->validateCpf($_POST['cpf'])) {
                throw new Exception('CPF inválido');
            }
            
            // Verificar se CPF já existe
            if ($this->employeeModel->cpfExists($_POST['cpf'])) {
                throw new Exception('CPF já cadastrado');
            }
            
            // Validar salário
            if (!is_numeric($_POST['salary']) || $_POST['salary'] <= 0) {
                throw new Exception('Salário deve ser um valor positivo');
            }
            
            // Preparar dados
            $data = [
                'full_name' => trim($_POST['full_name']),
                'cpf' => $_POST['cpf'],
                'rg' => trim($_POST['rg']),
                'birth_date' => $_POST['birth_date'],
                'address' => trim($_POST['address']),
                'phone_number' => trim($_POST['phone_number']),
                'job_title' => trim($_POST['job_title']),
                'salary' => floatval($_POST['salary']),
                'hire_date' => $_POST['hire_date'],
                'contract_type' => $_POST['contract_type'] ?? 'CLT',
                'cost_center' => trim($_POST['cost_center']) ?: null,
                'bank_name' => trim($_POST['bank_name']),
                'agency_number' => trim($_POST['agency_number']),
                'account_number' => trim($_POST['account_number']),
                'status' => $_POST['status'] ?? 'ativo'
            ];
            
            $employeeId = $this->employeeModel->create($data);
            
            if ($employeeId) {
                $_SESSION['success'] = 'Colaborador cadastrado com sucesso!';
                header('Location: ' . BASE_URL . '/employees/' . $employeeId);
            } else {
                throw new Exception('Erro ao cadastrar colaborador');
            }
            
        } catch (Exception $e) {
            $_SESSION['error'] = $e->getMessage();
            header('Location: ' . BASE_URL . '/employees/create');
        }
        
        exit;
    }
    
    /**
     * Exibir detalhes do colaborador
     */
    public function show($id) {
        AuthHelper::check();
        
        try {
            $employee = $this->employeeModel->findById($id);
            
            if (!$employee) {
                $_SESSION['error'] = 'Colaborador não encontrado';
                header('Location: ' . BASE_URL . '/employees');
                exit;
            }
            
            // Buscar documentos do colaborador
            $documents = $this->getEmployeeDocuments($id);
            
            require_once __DIR__ . '/../views/employees/show.php';
        } catch (Exception $e) {
            $_SESSION['error'] = 'Erro ao carregar colaborador: ' . $e->getMessage();
            header('Location: ' . BASE_URL . '/employees');
            exit;
        }
    }
    
    /**
     * Exibir formulário de edição
     */
    public function edit($id) {
        AuthHelper::check();
        
        try {
            $employee = $this->employeeModel->findById($id);
            
            if (!$employee) {
                $_SESSION['error'] = 'Colaborador não encontrado';
                header('Location: ' . BASE_URL . '/employees');
                exit;
            }
            
            require_once __DIR__ . '/../views/employees/edit.php';
        } catch (Exception $e) {
            $_SESSION['error'] = 'Erro ao carregar colaborador: ' . $e->getMessage();
            header('Location: ' . BASE_URL . '/employees');
            exit;
        }
    }
    
    /**
     * Processar atualização do colaborador
     */
    public function update($id) {
        AuthHelper::check();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: ' . BASE_URL . '/employees/' . $id . '/edit');
            exit;
        }
        
        try {
            // Verificar se colaborador existe
            $employee = $this->employeeModel->findById($id);
            if (!$employee) {
                throw new Exception('Colaborador não encontrado');
            }
            
            // Validar dados obrigatórios
            $requiredFields = [
                'full_name', 'cpf', 'rg', 'birth_date', 'address', 'phone_number',
                'job_title', 'salary', 'hire_date', 'bank_name', 'agency_number', 'account_number'
            ];
            
            foreach ($requiredFields as $field) {
                if (empty($_POST[$field])) {
                    throw new Exception("Campo {$field} é obrigatório");
                }
            }
            
            // Validar CPF
            if (!$this->employeeModel->validateCpf($_POST['cpf'])) {
                throw new Exception('CPF inválido');
            }
            
            // Verificar se CPF já existe (excluindo o próprio colaborador)
            if ($this->employeeModel->cpfExists($_POST['cpf'], $id)) {
                throw new Exception('CPF já cadastrado para outro colaborador');
            }
            
            // Validar salário
            if (!is_numeric($_POST['salary']) || $_POST['salary'] <= 0) {
                throw new Exception('Salário deve ser um valor positivo');
            }
            
            // Preparar dados
            $data = [
                'full_name' => trim($_POST['full_name']),
                'cpf' => $_POST['cpf'],
                'rg' => trim($_POST['rg']),
                'birth_date' => $_POST['birth_date'],
                'address' => trim($_POST['address']),
                'phone_number' => trim($_POST['phone_number']),
                'job_title' => trim($_POST['job_title']),
                'salary' => floatval($_POST['salary']),
                'hire_date' => $_POST['hire_date'],
                'contract_type' => $_POST['contract_type'] ?? 'CLT',
                'cost_center' => trim($_POST['cost_center']) ?: null,
                'bank_name' => trim($_POST['bank_name']),
                'agency_number' => trim($_POST['agency_number']),
                'account_number' => trim($_POST['account_number']),
                'status' => $_POST['status'] ?? 'ativo'
            ];
            
            $result = $this->employeeModel->update($id, $data);
            
            if ($result) {
                $_SESSION['success'] = 'Colaborador atualizado com sucesso!';
                header('Location: ' . BASE_URL . '/employees/' . $id);
            } else {
                throw new Exception('Erro ao atualizar colaborador');
            }
            
        } catch (Exception $e) {
            $_SESSION['error'] = $e->getMessage();
            header('Location: ' . BASE_URL . '/employees/' . $id . '/edit');
        }
        
        exit;
    }
    
    /**
     * Deletar colaborador
     */
    public function destroy($id) {
        AuthHelper::check();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: ' . BASE_URL . '/employees');
            exit;
        }
        
        try {
            // Verificar se colaborador existe
            $employee = $this->employeeModel->findById($id);
            if (!$employee) {
                throw new Exception('Colaborador não encontrado');
            }
            
            // Verificar se pode ser deletado (não tem registros dependentes)
            if ($this->hasRelatedRecords($id)) {
                throw new Exception('Não é possível excluir este colaborador pois possui registros relacionados. Considere inativá-lo.');
            }
            
            $result = $this->employeeModel->delete($id);
            
            if ($result) {
                $_SESSION['success'] = 'Colaborador excluído com sucesso!';
            } else {
                throw new Exception('Erro ao excluir colaborador');
            }
            
        } catch (Exception $e) {
            $_SESSION['error'] = $e->getMessage();
        }
        
        header('Location: ' . BASE_URL . '/employees');
        exit;
    }
    
    /**
     * Upload de documento para um colaborador
     */
    public function uploadDocument($id) {
        AuthHelper::check();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $_SESSION['error'] = 'Método não permitido';
            header('Location: ' . BASE_URL . '/employees/' . $id);
            exit;
        }
        
        try {
            // Verifica se o colaborador existe
            $employee = $this->employeeModel->findById($id);
            if (!$employee) {
                $_SESSION['error'] = 'Colaborador não encontrado';
                header('Location: ' . BASE_URL . '/employees');
                exit;
            }
            
            // Verifica se foi enviado um arquivo
            if (!isset($_FILES['document']) || empty($_FILES['document']['name'])) {
                $_SESSION['error'] = 'Nenhum arquivo foi selecionado';
                header('Location: ' . BASE_URL . '/employees/' . $id);
                exit;
            }
            
            $file = $_FILES['document'];
            
            // Valida o arquivo
            require_once __DIR__ . '/../models/EmployeeDocument.php';
            $errors = EmployeeDocument::validateFile($file);
            
            if (!empty($errors)) {
                $_SESSION['error'] = implode(', ', $errors);
                header('Location: ' . BASE_URL . '/employees/' . $id);
                exit;
            }
            
            // Cria o diretório do colaborador
            $employeeDir = EmployeeDocument::createEmployeeDirectory($id);
            
            // Gera nome único para o arquivo
            $fileName = EmployeeDocument::generateFileName($file['name']);
            $filePath = $employeeDir . '/' . $fileName;
            
            // Move o arquivo para o diretório
            if (!move_uploaded_file($file['tmp_name'], $filePath)) {
                throw new Exception('Erro ao salvar o arquivo');
            }
            
            // Salva no banco de dados
            $documentModel = new EmployeeDocument();
            $finfo = finfo_open(FILEINFO_MIME_TYPE);
            $mimeType = finfo_file($finfo, $filePath);
            finfo_close($finfo);
            
            $documentModel->create(
                $id,
                $file['name'],
                $fileName,
                $filePath,
                $file['size'],
                $mimeType
            );
            
            $_SESSION['success'] = 'Documento enviado com sucesso!';
            
        } catch (Exception $e) {
            $_SESSION['error'] = 'Erro ao enviar documento: ' . $e->getMessage();
        }
        
        header('Location: ' . BASE_URL . '/employees/' . $id);
    }
    
    /**
     * Download de documento de um colaborador
     */
    public function downloadDocument($employeeId, $documentId) {
        AuthHelper::check();
        
        try {
            require_once __DIR__ . '/../models/EmployeeDocument.php';
            $documentModel = new EmployeeDocument();
            
            // Busca o documento
            $document = $documentModel->getByIdAndEmployeeId($documentId, $employeeId);
            
            if (!$document) {
                $_SESSION['error'] = 'Documento não encontrado';
                header('Location: ' . BASE_URL . '/employees/' . $employeeId);
                exit;
            }
            
            // Verifica se o arquivo existe
            if (!file_exists($document['file_path'])) {
                $_SESSION['error'] = 'Arquivo não encontrado no servidor';
                header('Location: ' . BASE_URL . '/employees/' . $employeeId);
                exit;
            }
            
            // Define os headers para download
            header('Content-Type: ' . $document['mime_type']);
            header('Content-Disposition: attachment; filename="' . $document['original_name'] . '"');
            header('Content-Length: ' . $document['file_size']);
            header('Cache-Control: no-cache, must-revalidate');
            header('Pragma: no-cache');
            
            // Envia o arquivo
            readfile($document['file_path']);
            exit;
            
        } catch (Exception $e) {
            $_SESSION['error'] = 'Erro ao baixar documento: ' . $e->getMessage();
            header('Location: ' . BASE_URL . '/employees/' . $employeeId);
        }
    }
    
    /**
     * Exclusão de documento de um colaborador
     */
    public function deleteDocument($employeeId, $documentId) {
        AuthHelper::check();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $_SESSION['error'] = 'Método não permitido';
            header('Location: ' . BASE_URL . '/employees/' . $employeeId);
            exit;
        }
        
        try {
            require_once __DIR__ . '/../models/EmployeeDocument.php';
            $documentModel = new EmployeeDocument();
            
            // Busca o documento
            $document = $documentModel->getByIdAndEmployeeId($documentId, $employeeId);
            
            if (!$document) {
                $_SESSION['error'] = 'Documento não encontrado';
                header('Location: ' . BASE_URL . '/employees/' . $employeeId);
                exit;
            }
            
            // Remove o arquivo físico
            if (file_exists($document['file_path'])) {
                unlink($document['file_path']);
            }
            
            // Remove do banco de dados
            $documentModel->delete($documentId);
            
            $_SESSION['success'] = 'Documento excluído com sucesso!';
            
        } catch (Exception $e) {
            $_SESSION['error'] = 'Erro ao excluir documento: ' . $e->getMessage();
        }
        
        header('Location: ' . BASE_URL . '/employees/' . $employeeId);
    }
    
    /**
     * Métodos auxiliares
     */
    private function hasRelatedRecords($employeeId) {
        // Verificar se tem registros de ponto, ausências ou folhas de pagamento
        try {
            $pdo = new PDO(
                "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME,
                DB_USER,
                DB_PASS,
                [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
            );
            
            $tables = ['time_records', 'absences', 'payrolls'];
            
            foreach ($tables as $table) {
                $stmt = $pdo->prepare("SELECT COUNT(*) FROM {$table} WHERE employee_id = ?");
                $stmt->execute([$employeeId]);
                
                if ($stmt->fetchColumn() > 0) {
                    return true;
                }
            }
            
            return false;
        } catch (PDOException $e) {
            return true; // Em caso de erro, não permitir exclusão
        }
    }
    
    private function getEmployeeDocuments($employeeId) {
        try {
            $pdo = new PDO(
                "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME,
                DB_USER,
                DB_PASS,
                [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
            );
            
            $stmt = $pdo->prepare("SELECT * FROM employee_documents WHERE employee_id = ? ORDER BY uploaded_at DESC");
            $stmt->execute([$employeeId]);
            
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            return [];
        }
    }
    
    private function getEmployeeDocument($documentId) {
        try {
            $pdo = new PDO(
                "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME,
                DB_USER,
                DB_PASS,
                [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
            );
            
            $stmt = $pdo->prepare("SELECT * FROM employee_documents WHERE id = ?");
            $stmt->execute([$documentId]);
            
            return $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            return false;
        }
    }
    
    private function saveEmployeeDocument($employeeId, $documentName, $originalFilename, $fileName, $mimeType, $fileSize) {
        try {
            $pdo = new PDO(
                "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME,
                DB_USER,
                DB_PASS,
                [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
            );
            
            $stmt = $pdo->prepare("
                INSERT INTO employee_documents 
                (employee_id, document_name, original_filename, file_path, file_size, mime_type) 
                VALUES (?, ?, ?, ?, ?, ?)
            ");
            
            return $stmt->execute([
                $employeeId,
                $documentName,
                $originalFilename,
                $fileName,
                $fileSize,
                $mimeType
            ]);
        } catch (PDOException $e) {
            throw new Exception('Erro ao salvar informações do documento: ' . $e->getMessage());
        }
    }
}