<?php

class Conta
{
    private $pdo;

    public function __construct()
    {
        try {
            $this->pdo = new PDO('mysql:host=' . DB_HOST . ';dbname=' . DB_NAME, DB_USER, DB_PASS);
            $this->pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            $this->ensureSchema();
        } catch (PDOException $e) {
            die("Erro na conexão com o banco de dados: " . $e->getMessage());
        }
    }

    private function ensureSchema()
    {
        $sql = "CREATE TABLE IF NOT EXISTS transacoes (
            id INT NOT NULL AUTO_INCREMENT,
            data_transacao DATE NOT NULL,
            tipo ENUM('Receita','Despesa') NOT NULL,
            descricao VARCHAR(255) NOT NULL,
            valor DECIMAL(10,2) NOT NULL,
            forma_pagamento VARCHAR(100) NULL,
            comprovante_pagamento VARCHAR(255) NULL,
            entidade VARCHAR(255) NULL,
            origem VARCHAR(100) NOT NULL,
            data_cadastro TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci";
        $this->pdo->exec($sql);
    }

    public function getAll($ano, $mes)
    {
        $sql = "SELECT * FROM transacoes WHERE YEAR(data_transacao) = :ano";
        $params = [':ano' => $ano];

        if ($mes != 0) {
            $sql .= " AND MONTH(data_transacao) = :mes";
            $params[':mes'] = $mes;
        }

        $sql .= " ORDER BY data_transacao DESC, id DESC";

        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getResumo($ano, $mes)
    {
        $sql = "SELECT 
                    SUM(CASE WHEN tipo = 'Receita' THEN valor ELSE 0 END) as total_receitas,
                    SUM(CASE WHEN tipo = 'Despesa' THEN valor ELSE 0 END) as total_despesas
                FROM transacoes 
                WHERE YEAR(data_transacao) = :ano";

        $params = [':ano' => $ano];

        if ($mes != 0) {
            $sql .= " AND MONTH(data_transacao) = :mes";
            $params[':mes'] = $mes;
        }

        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function create($dados)
    {
        $sql = "INSERT INTO transacoes (
                    data_transacao, tipo, descricao, valor, 
                    forma_pagamento, comprovante_pagamento, entidade, origem
                ) VALUES (
                    :data_transacao, :tipo, :descricao, :valor, 
                    :forma_pagamento, :comprovante_pagamento, :entidade, :origem
                )";

        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([
            ':data_transacao' => $dados['data_transacao'],
            ':tipo' => $dados['tipo'],
            ':descricao' => $dados['descricao'],
            ':valor' => $dados['valor'],
            ':forma_pagamento' => $dados['forma_pagamento'] ?? null,
            ':comprovante_pagamento' => $dados['comprovante_pagamento'] ?? null,
            ':entidade' => $dados['entidade'] ?? null,
            ':origem' => $dados['origem']
        ]);

        return $this->pdo->lastInsertId();
    }

    public function delete($id)
    {
        // Primeiro busca para pegar o nome do arquivo e deletar se existir
        $stmt = $this->pdo->prepare("SELECT comprovante_pagamento FROM transacoes WHERE id = :id");
        $stmt->execute([':id' => $id]);
        $transacao = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($transacao && $transacao['comprovante_pagamento']) {
            $filePath = __DIR__ . '/../../public_html/contas/uploads/comprovantes/' . $transacao['comprovante_pagamento'];
            // Ajuste de caminho: o upload original estava em public_html/contas/uploads
            // Vamos manter compatibilidade ou mover para um lugar novo?
            // Por enquanto, vamos assumir que o controller lida com o arquivo físico ou ajustamos aqui.
            // Melhor deixar o controller lidar com arquivo físico para não acoplar Model com FileSystem demais.
        }

        $sql = "DELETE FROM transacoes WHERE id = :id";
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute([':id' => $id]);
    }

    public function findById($id)
    {
        $stmt = $this->pdo->prepare("SELECT * FROM transacoes WHERE id = :id");
        $stmt->execute([':id' => $id]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
}
