<?php

class EmployeeDocument {
    private $pdo;
    
    public function __construct() {
        try {
            $this->pdo = new PDO(
                "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4",
                DB_USER,
                DB_PASS,
                [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                    PDO::ATTR_EMULATE_PREPARES => false
                ]
            );
        } catch (PDOException $e) {
            throw new Exception("Erro na conexão com o banco de dados: " . $e->getMessage());
        }
    }
    
    /**
     * Salva um documento no banco de dados
     */
    public function create($employeeId, $originalName, $fileName, $filePath, $fileSize, $mimeType) {
        $sql = "INSERT INTO employee_documents (employee_id, original_name, file_name, file_path, file_size, mime_type, uploaded_at) 
                VALUES (?, ?, ?, ?, ?, ?, NOW())";
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([$employeeId, $originalName, $fileName, $filePath, $fileSize, $mimeType]);
        
        return $this->pdo->lastInsertId();
    }
    
    /**
     * Busca todos os documentos de um colaborador
     */
    public function getByEmployeeId($employeeId) {
        $sql = "SELECT * FROM employee_documents WHERE employee_id = ? ORDER BY uploaded_at DESC";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([$employeeId]);
        
        return $stmt->fetchAll();
    }
    
    /**
     * Busca um documento específico
     */
    public function getById($id) {
        $sql = "SELECT * FROM employee_documents WHERE id = ?";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([$id]);
        
        return $stmt->fetch();
    }
    
    /**
     * Busca um documento específico de um colaborador
     */
    public function getByIdAndEmployeeId($id, $employeeId) {
        $sql = "SELECT * FROM employee_documents WHERE id = ? AND employee_id = ?";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([$id, $employeeId]);
        
        return $stmt->fetch();
    }
    
    /**
     * Remove um documento
     */
    public function delete($id) {
        // Primeiro busca o documento para obter o caminho do arquivo
        $document = $this->getById($id);
        if (!$document) {
            return false;
        }
        
        // Remove o arquivo físico
        if (file_exists($document['file_path'])) {
            unlink($document['file_path']);
        }
        
        // Remove do banco de dados
        $sql = "DELETE FROM employee_documents WHERE id = ?";
        $stmt = $this->pdo->prepare($sql);
        
        return $stmt->execute([$id]);
    }
    
    /**
     * Remove todos os documentos de um colaborador
     */
    public function deleteByEmployeeId($employeeId) {
        $documents = $this->getByEmployeeId($employeeId);
        
        foreach ($documents as $document) {
            if (file_exists($document['file_path'])) {
                unlink($document['file_path']);
            }
        }
        
        $sql = "DELETE FROM employee_documents WHERE employee_id = ?";
        $stmt = $this->pdo->prepare($sql);
        
        return $stmt->execute([$employeeId]);
    }
    
    /**
     * Conta o número de documentos de um colaborador
     */
    public function countByEmployeeId($employeeId) {
        $sql = "SELECT COUNT(*) as total FROM employee_documents WHERE employee_id = ?";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([$employeeId]);
        
        $result = $stmt->fetch();
        return $result['total'];
    }
    
    /**
     * Valida se o arquivo é permitido
     */
    public static function validateFile($file) {
        $errors = [];
        
        // Verifica se houve erro no upload
        if ($file['error'] !== UPLOAD_ERR_OK) {
            switch ($file['error']) {
                case UPLOAD_ERR_INI_SIZE:
                case UPLOAD_ERR_FORM_SIZE:
                    $errors[] = 'O arquivo é muito grande. Tamanho máximo: 5MB';
                    break;
                case UPLOAD_ERR_PARTIAL:
                    $errors[] = 'O upload do arquivo foi interrompido';
                    break;
                case UPLOAD_ERR_NO_FILE:
                    $errors[] = 'Nenhum arquivo foi selecionado';
                    break;
                default:
                    $errors[] = 'Erro no upload do arquivo';
            }
            return $errors;
        }
        
        // Verifica o tamanho do arquivo (5MB máximo)
        $maxSize = 5 * 1024 * 1024; // 5MB em bytes
        if ($file['size'] > $maxSize) {
            $errors[] = 'O arquivo é muito grande. Tamanho máximo: 5MB';
        }
        
        // Verifica o tipo do arquivo
        $allowedTypes = [
            'application/pdf',
            'image/jpeg',
            'image/jpg', 
            'image/png'
        ];
        
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mimeType = finfo_file($finfo, $file['tmp_name']);
        finfo_close($finfo);
        
        if (!in_array($mimeType, $allowedTypes)) {
            $errors[] = 'Tipo de arquivo não permitido. Apenas PDF, JPG e PNG são aceitos';
        }
        
        // Verifica a extensão do arquivo
        $allowedExtensions = ['pdf', 'jpg', 'jpeg', 'png'];
        $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        
        if (!in_array($extension, $allowedExtensions)) {
            $errors[] = 'Extensão de arquivo não permitida';
        }
        
        return $errors;
    }
    
    /**
     * Gera um nome único para o arquivo
     */
    public static function generateFileName($originalName) {
        $extension = strtolower(pathinfo($originalName, PATHINFO_EXTENSION));
        $timestamp = time();
        $random = bin2hex(random_bytes(8));
        
        return $timestamp . '_' . $random . '.' . $extension;
    }
    
    /**
     * Cria o diretório do colaborador se não existir
     */
    public static function createEmployeeDirectory($employeeId) {
        $baseDir = __DIR__ . '/../../storage/documents';
        $employeeDir = $baseDir . '/' . $employeeId;
        
        if (!is_dir($employeeDir)) {
            if (!mkdir($employeeDir, 0755, true)) {
                throw new Exception('Não foi possível criar o diretório para os documentos');
            }
        }
        
        return $employeeDir;
    }
    
    /**
     * Formata o tamanho do arquivo para exibição
     */
    public static function formatFileSize($bytes) {
        if ($bytes >= 1073741824) {
            return number_format($bytes / 1073741824, 2) . ' GB';
        } elseif ($bytes >= 1048576) {
            return number_format($bytes / 1048576, 2) . ' MB';
        } elseif ($bytes >= 1024) {
            return number_format($bytes / 1024, 2) . ' KB';
        } else {
            return $bytes . ' bytes';
        }
    }
    
    /**
     * Obtém o ícone do arquivo baseado no tipo MIME
     */
    public static function getFileIcon($mimeType) {
        switch ($mimeType) {
            case 'application/pdf':
                return '📄';
            case 'image/jpeg':
            case 'image/jpg':
            case 'image/png':
                return '🖼️';
            default:
                return '📎';
        }
    }
}