<?php
require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../database/connection.php';
require_once __DIR__ . '/../../helpers/auth_helper.php';

// Verificar se o usuário está autenticado
check_auth();

// Verificar se o método é POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    $_SESSION['error'] = 'Método inválido';
    header('Location: ' . BASE_URL . '/employees');
    exit;
}

// Verificar se o ID do colaborador foi fornecido
if (!isset($_POST['employee_id']) || empty($_POST['employee_id'])) {
    $_SESSION['error'] = 'ID do colaborador não fornecido';
    header('Location: ' . BASE_URL . '/employees');
    exit;
}

$employee_id = filter_input(INPUT_POST, 'employee_id', FILTER_SANITIZE_NUMBER_INT);

// Verificar se o arquivo foi enviado
if (!isset($_FILES['photo']) || $_FILES['photo']['error'] === UPLOAD_ERR_NO_FILE) {
    $_SESSION['error'] = 'Nenhuma foto foi selecionada';
    header('Location: ' . BASE_URL . '/employees/show.php?id=' . $employee_id);
    exit;
}

// Verificar se houve erro no upload
if ($_FILES['photo']['error'] !== UPLOAD_ERR_OK) {
    $error_messages = [
        UPLOAD_ERR_INI_SIZE => 'O arquivo excede o tamanho máximo permitido pelo servidor',
        UPLOAD_ERR_FORM_SIZE => 'O arquivo excede o tamanho máximo permitido pelo formulário',
        UPLOAD_ERR_PARTIAL => 'O upload do arquivo foi feito parcialmente',
        UPLOAD_ERR_NO_TMP_DIR => 'Pasta temporária ausente',
        UPLOAD_ERR_CANT_WRITE => 'Falha ao escrever arquivo em disco',
        UPLOAD_ERR_EXTENSION => 'Uma extensão PHP interrompeu o upload do arquivo'
    ];
    
    $error_code = $_FILES['photo']['error'];
    $error_message = isset($error_messages[$error_code]) ? $error_messages[$error_code] : 'Erro desconhecido no upload';
    
    $_SESSION['error'] = $error_message;
    header('Location: ' . BASE_URL . '/employees/show.php?id=' . $employee_id);
    exit;
}

// Verificar o tipo de arquivo
$allowed_types = ['image/jpeg', 'image/png', 'image/gif'];
$file_type = $_FILES['photo']['type'];

if (!in_array($file_type, $allowed_types)) {
    $_SESSION['error'] = 'Tipo de arquivo não permitido. Apenas JPG, PNG e GIF são aceitos';
    header('Location: ' . BASE_URL . '/employees/show.php?id=' . $employee_id);
    exit;
}

// Verificar o tamanho do arquivo (máximo 2MB)
$max_size = 2 * 1024 * 1024; // 2MB em bytes
if ($_FILES['photo']['size'] > $max_size) {
    $_SESSION['error'] = 'O arquivo excede o tamanho máximo de 2MB';
    header('Location: ' . BASE_URL . '/employees/show.php?id=' . $employee_id);
    exit;
}

// Criar diretório de upload se não existir
$upload_dir = $_SERVER['DOCUMENT_ROOT'] . '/gabinete/uploads/employees/';
if (!is_dir($upload_dir)) {
    mkdir($upload_dir, 0755, true);
}

// Gerar nome único para o arquivo
$file_extension = pathinfo($_FILES['photo']['name'], PATHINFO_EXTENSION);
$new_filename = uniqid('employee_') . '.' . $file_extension;
$upload_path = $upload_dir . $new_filename;

// Mover o arquivo para o diretório de upload
if (!move_uploaded_file($_FILES['photo']['tmp_name'], $upload_path)) {
    $_SESSION['error'] = 'Falha ao mover o arquivo enviado';
    header('Location: ' . BASE_URL . '/employees/show.php?id=' . $employee_id);
    exit;
}

// Atualizar o registro do colaborador no banco de dados
try {
    $pdo = get_db_connection();
    
    // Primeiro, obter o nome da foto atual (se existir)
    $stmt = $pdo->prepare("SELECT photo FROM employees WHERE id = :id");
    $stmt->bindParam(':id', $employee_id, PDO::PARAM_INT);
    $stmt->execute();
    $employee = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Se existir uma foto anterior, excluí-la
    if (!empty($employee['photo']) && file_exists($upload_dir . $employee['photo'])) {
        unlink($upload_dir . $employee['photo']);
    }
    
    // Atualizar o registro com o novo nome de arquivo
    $stmt = $pdo->prepare("UPDATE employees SET photo = :photo WHERE id = :id");
    $stmt->bindParam(':photo', $new_filename, PDO::PARAM_STR);
    $stmt->bindParam(':id', $employee_id, PDO::PARAM_INT);
    $stmt->execute();
    
    $_SESSION['success'] = 'Foto do colaborador atualizada com sucesso';
} catch (PDOException $e) {
    // Em caso de erro no banco de dados, excluir o arquivo enviado
    if (file_exists($upload_path)) {
        unlink($upload_path);
    }
    
    $_SESSION['error'] = 'Erro ao atualizar a foto do colaborador: ' . $e->getMessage();
}

// Redirecionar de volta para a página do colaborador
header('Location: ' . BASE_URL . '/employees/show.php?id=' . $employee_id);
exit;